/**
 * Halcyon: Hal's C library
 * (c) Hal Vaughan 2008
 * hal@halblog.com
 * licensed under the Free Software Foundations General Public License 2.0
 *
 * This is all or part of my C/C++ functions library that may be used as part
 * of other open source programs.
 */

#include <cstdlib>
#include <string>
#include <iostream>
#include <fstream>
#include <map>
#include <vector>
#include <sys/stat.h>
#include "halcyon.h"
using namespace std;

//Leave these comments in place -- the original author uses them with a
//Perl script to generate headers

//ConfigFile::	bool verbose, argsparsed, configparsed;
//ConfigFile::	unsigned int cmdpoint;
//ConfigFile::	string configFile;
//ConfigFile::	map<string, string> configVals;
//ConfigFile::	vector<string> cmdvals;
//ConfigFile::	halcyon utillib;

/**
 * Trim a string, generally used in reading data from config files.  Remove the
 * whitespace and quotation marks (single and double) from the start of a string.
 * @param istr pointer to the string to be trimmed.
 */
void halcyon::trimstringstart(string* istr) { //public
	bool first = true;
	char c;
	int x = -1, iLimit = istr->size();
	for (x = 0; x < iLimit; x++) {
		c = (*istr)[x];
		if (c != ' ' && c != '\t' && c != '\'' && c != '"')
			break;
		first = false;
	}
	if (first)
		x--;
	if (x > 0)
		istr->erase(0, x);
	return;
}

/**
 * Trim a string, generally used in reading data from config files.  Remove the
 * whitespace and quotation marks (single and double) from the end of a string.
 * @param istr pointer to the string to be trimmed.
 */
void halcyon::trimstringend(string* istr) { //public
	char c;
	int x, ic = -1, iLimit = istr->size();
	for (x = iLimit - 1; x > -1; x--) {
		c = (*istr)[x];
		if (c != ' ' && c != '\t' && c != '\'' && c != '"')
			break;
		ic++;
	}
	x++;
	ic++;
	if (ic > 0)
		istr->erase(x, ic);
	return;
}

/**
 * Short cut to call both trimstringstart() and trimstringend() to trim off the
 * whitespace and quotation marks (both single and double) from the beginning and
 * end of a string.
 */
void halcyon::trimstring(string* istr) { //public
	trimstringstart(istr);
	trimstringend(istr);
	return;
}

/**
 * Create a ConfigFile object.  A ConfigFile can read in a config file made up of key=value pairs
 * and parse the lines, including trimming the quotation marks (single and double) and whitespace
 * out of the way.  It then keeps the key/value pairs stored so they can be easily accessed by a
 * program.  Command line arguments can also be passed in.  The intent is to read in a config
 * file, then read the command line args so they can override settings in the config file.  It is
 * also possible to read in a sequence of config files, with each one overriding the settings of
 * the previous.  If a program has a system config file in /etc, then other config files for
 * users, then the system one could be read first, then the user files so each user is able to
 * have their own settings.  We ignore any blank lines or lines starting with // or #.  We do
 * not store them, though, so they are not written back out if we write the config file back out.
 *
 * Also commands can be stored from the command line arguments, since many programs can act on
 * them.  The difference is that settings have "--" in front of them and commands don't.  Any argument
 * parsed from the command line without the double-dash in front of it will be stored as a command so
 * the program can read them in order and act on them if desired.
 *
 * Files are also written back out, with the values in alphabetical order (a feature of
 * map<string,string>, not of this class!) to one or more config files.
 */
ConfigFile::ConfigFile () { //public
	verbose = false;
	argsparsed = false;
	configparsed = false;
	string configFile = "";
}

/**
 * Parse the arguments from the command line as if they were actually in
 * a config file and store their values.  This can be called directly from main()
 * with argc and argv.
 * @param count same as argc
 * @param args same as argv
 */
void ConfigFile::parseargs(int count, char** args) { //public
	int x;
	string arg, key, val;

	argsparsed = true;
	for (x = 1; x < count; x++) {
		arg = args[x];
		parseline(arg);
	}
	return;
}

/**
 * Parse a line of data and store the values.  This can be an argument from
 * the command line or a line of data in a config file.
 */
void ConfigFile::parseline(string arg) { //public
	int i;
	string key, val;

	utillib.trimstring(&arg);
	if (arg.compare(0,2, "--") == 0) {
		i = arg.find_first_of("=");
		if (i < 0) {
			key = arg.substr(2, arg.size() - 1);
			val = "true";
		} else {
			key = arg.substr(2, i - 2);
			val = arg.substr(i + 1, arg.size() - i);
			utillib.trimstring(&val);
		}
		utillib.trimstring(&key);
		setval(key, val);
		if (key == "config") {
			configFile = val;
			loadconfig();
		}
	}  else {
		cmdvals.push_back(arg);
		if (verbose) cout << "Addint command to queue: " << arg << endl;
	}
	return;
}

/**
 * Set a value.  Since no argument is give, it is taken that the value needs to
 * be specified, so it will be set as true.
 * @param key name of value to set to true.
 */
void ConfigFile::setval(string key) { //public
	setval(key, "true");
	return;
}

/**
 * Set a variable to a specified value.
 * @param key name of variable to set
 * @param val value to assign to the variable
 */
void ConfigFile::setval(string key, string val) { //public
	if (verbose) cout << "Setting config value, Key: " << key << ", Value: " << val << endl;
	configVals[key] = val;
	return;
}

/**
 * Set a variable to an integer value.  Since all values are stored as strings, the
 * int will be converted to a string for storage.
 * @param key name of variable
 * @param ival value to assign to variable.
 */
void ConfigFile::setval(string key, int ival) { //public
	char xfer[10];
	sprintf(xfer, "%d", ival);
	string sval = xfer;
	setval(key, sval);
	return;
}

/**
 * Set a value to a boolean.
 * @param key name of variable
 * @param bval value to set variable to
 */
void ConfigFile::setboolval(string key, bool bval) { //public
	string sval = "false";
	if (bval)
		sval = "true";
	setval(key, sval);
	return;
}

/**
 * Unset or remove a variable so it's not stored or written back out
 * to a config file.
 * @param key name of variable to unset
 */
void ConfigFile::unset(string key) {//public
	map<string,string>::iterator it = configVals.find(key);
	configVals.erase(it);
	return;
}

/**
 * Get the value for a variable.
 * @param key name of variable to get value for
 * @return the value of that variable
 */
string ConfigFile::getval(string key) { //public
	return configVals[key];
}

/**
 * Get an integer value of a variable.  Does not do any checking to be sure
 * the variable is an int when converting it from a string.
 * @param key the variable to get the value of
 * @return the int value of that variable
 */
int ConfigFile::getintval(string key) { //public
	int i = -1;
	string val = configVals[key];
	if (val == "")
		return i;
	sscanf(val.c_str(), "%d", &i);
	return i;
}

/**
 * Get the boolean value of a variable.  Does not do any checking to be sure
 * the variable is a boolean when converting it from a string.
 * @param key the variable to get the value of
 * @return the boolean value of that variable
 */
bool ConfigFile::getboolval(string key) { //public
	bool bval = false;
	string val = configVals[key];
	if (val == "true")
		bval = true;
	return bval;
}

/**
 * Specify the file to use as a config file for reading and writing.
 * @param cfname name of config file
 */
void ConfigFile::setconfigfile(string cfname) { //public
	configFile = cfname;
	return;
}

/**
 * Set the verbosity level for debugging output.
 * @param verbosity true for debugging output
 */
void ConfigFile::setverbose(bool verbosity) { //public
	verbose = verbosity;
	return;
}

/**
 * Read in the already set config file and parse it.
 */
void ConfigFile::loadconfig() { //public
	int fstat, isplit;
	string fline, key, val;
	ifstream cfile;
	struct stat finfo;

// 	verbose = true;

	if (configFile == "") {
		if (verbose) cout <<  "Config file name is blank.\n";
		return;
	}
	fstat = stat(configFile.c_str(), &finfo);
	if (fstat < 0) {
		if (verbose) cout << "Config file does not exist.\n";
		return;
	}
	cfile.open(configFile.c_str(), iostream::in);
	if (!cfile) {
		if (verbose) cout << "Config file could not be opened.\n";
		return;
	}
	if (verbose) cout << "Reading in config file: " << configFile << endl;
	while (!cfile.eof()) {
		getline(cfile, fline);
		if (fline.size() == 0 || fline.substr(0,1) == "#" || fline.substr(0,2) == "//" || fline == "\n")
			continue;
// 		cout << "Line to clean: ->" << fline << "<-\n";
		utillib.trimstring(&fline);
// 		cout << "Line to use: ->" << fline << "<-\n";
		if (fline.size() == 0 || fline.substr(0,1) == "#" || fline.substr(0,2) == "//")
			continue;
		isplit = fline.find("=", 0);
		if (isplit > 0) {
			key = fline.substr(0, isplit);
			val = fline.substr(isplit + 1, fline.size() - isplit);
			utillib.trimstring(&key);
			utillib.trimstring(&val);
		} else {
			key = fline;
			utillib.trimstring(&key);
			val = "true";
		}
//If we're parsing the command line, don't override an existing setting (command line takes priority)
		if (argsparsed) {
			fline = configVals[key];
			if (fline != "")
				continue;
		}
		setval(key, val);
	}
	cfile.close();
	configparsed = true;
	return;
}

/**
 * Load the specified config file and parse it.
 * @param cfname name of config file to load
 */
void ConfigFile::loadconfig(string cfname) { //public
	setconfigfile(cfname);
	loadconfig();
	return;
}

/**
 * Save the config file to the given file name.  This clobbers any existing
 * file of the same name.
 * @param cfname name of the config file we write to.
 * @return true if the file was written to okay
 */
bool ConfigFile::saveconfig(string cfname) {//public
	setconfigfile(cfname);
	bool didwrite = saveconfig();
	return didwrite;
}

/**
 * Save the config file to the current config file name.
 * @return false if the file could not be written.
 */
bool ConfigFile::saveconfig() {//public
	bool wrotefile = true;
	string key, val;
	map<string,string>::iterator it;
	ofstream outfile (configFile.c_str());

	if (verbose) cout << "Writing out config file: " << configFile << endl;
	setboolval("firstrun", false);
	if (outfile.is_open()) {
		for (it = configVals.begin(); it != configVals.end(); it++) {
			key = (*it).first;
			val = (*it).second;
			outfile << key << "=" << val << endl;
		}
	} else {
		wrotefile = false;
	}
	return wrotefile;
}

/**
 * Get all the commands passed on to us in a single vector.
 * @return vector of all the commands in the queue.
 */
vector<string> ConfigFile::getcommands() { //public
	return cmdvals;
}

/**
 * Get the size of the command queue.
 * @return the number of commands in the queue
 */
int ConfigFile::getcommandsize() { //public
	return cmdvals.size();
}

/**
 * Get the next command in the command queue.
 * @return the next command
 */
string ConfigFile::getnextcommand() { //public
	string cmd = "";
	if (cmdpoint < cmdvals.size())
		cmd = cmdvals[cmdpoint];
	return cmd;
}

