package com.hal.conf;
/**
 * Program: Many different FOSS (Free/Open Source Software) Programs
 * Copyright (C) 2007 by Hal Vaughan, hal@halblog.com  
 * Licensed under the General Public License (GPL) Version 2.0
 * See http://halblog.com for more information.
 */
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;

import com.hal.util.HalFile;
import com.hal.util.StringHashMap;


public class GeneralConfig {
		
	public String configFile = "";
	
	public String configFileParameter = "Config";
	
	private StringHashMap mainSettings = new StringHashMap(), cliCommands = new StringHashMap();
	
	private HashMap globalObjects = new HashMap();
	
	public GeneralConfig(String configFileName) {
		configFile = configFileName;
		mainSettings = HalFile.fileToHash(configFile);
		cliCommands = new StringHashMap();
	}
	
	/**
	 * Create a config object.  Go through all the items in the array.  If they start with &quot;--&quot; then they're
	 * commands and we put them in a hash and set them as true unless some other value was passed in the command (after
	 * the equals sign).  Also store all the config values in a hash as well.  Then read in the config file if we're given one.
	 * @param originalArgs
	 */
	public GeneralConfig(String[] originalArgs) {
		int x, y;
		String sArg, sKey, sVal;
		String[] aList;
		for (x = 0; x < originalArgs.length; x++) {
			sArg = originalArgs[x];
//			System.out.println("Reading config parm: " + sArg);
			if (sArg.startsWith("-")) {
				while (sArg.startsWith("-")) {
					sArg = sArg.substring(1);
				}
				if (sArg.contains("=")) {
					aList = sArg.split("=");
					sKey = aList[0];
					sVal = aList[1];
				} else {
					sKey = sArg;
					sVal = "true";
				}
				setCommand(sKey, sVal);
			} else {
				aList = sArg.split("=");
				sKey = aList[0];
				sVal = "";
				for (y = 1; y < aList.length; y++) {
					sVal = sVal + "=" + aList[y];
				}
				if (sVal.length() > 1) {
					sVal = sVal.substring(1);
				} else {
					sVal = "";
				}
				set(sKey, sVal);
				if (sKey.equalsIgnoreCase(configFileParameter)) {
//					System.out.println("Reading config file: " + sVal);
					readConfigFile(sVal);
				}
			}
		}
	}
	
	/**
	 * Read in the config file and store the values as settings
	 * @param configFileName name of config file
	 */
	private void readConfigFile(String configFileName) {
		configFile = configFileName;
		mainSettings = HalFile.fileToHash(configFile);
		set(configFileParameter, configFile);
		return;
	}
	
	/**
	 * Save the config file.  There should only be one config file per program and if it
	 * has been changed, we will want to be able to save it to keep the changes.
	 */
	public void saveConfigFile() {
		int x;
		String sLine;
		String[] allKeys = mainSettings.keySet();
		HalFile fileWriter = new HalFile();
		fileWriter.openTextFileOut(configFile);
		for (x = 0; x < allKeys.length; x++) {
			sLine = allKeys[x] + "=" + mainSettings.get(allKeys[x]) + "\n";
			fileWriter.writeTextOut(sLine);
		}
		fileWriter.closeTextFileOut();
		return;
	}
	
	/**
	 * Get the value of a command.  In many cases it's only true or false, but in
	 * a string value if retrieved from here.  Other times it's a value for a command
	 * line option
	 * @param commandName name of command
	 * @return value to go with the command
	 */
	public String getCommand(String commandName) {
		String sVal = "";
		if (cliCommands.hasKey(commandName)) {
			sVal = cliCommands.get(commandName);
			if (sVal == null) {sVal = "";}
		}
		return sVal;
	}
	
	/**
	 * Get a command value, but only in boolean.  Command values are set on the command line
	 * with a -- in front of the command (settings don't have the -- in front).
	 * @param commandName name of command
	 * @return value of command
	 */
	public boolean getTFCommand(String commandName) {
		boolean bFlag = false;
		String sVal = "";
		if (cliCommands.hasKey(commandName)) {
			sVal = cliCommands.get(commandName);
			if (sVal == null) sVal = "false";
			if (sVal.equalsIgnoreCase("true"))
				bFlag = true;
		}
		return bFlag;
	}
	
	/**
	 * Set the value for a command.  Sometimes the inital class will need to set a command
	 * value from what it gets from teh command line.  This simply sets it to true.
	 * @param commandName name of command
	 */
	public void setCommand(String commandName) {
		cliCommands.put(commandName, "true");
		return;
	}
	
	/**
	 * Set a command value other than to a default true.
	 * @param commandName command to set
	 * @param commandValue value to set command to
	 */
	public void setCommand(String commandName, String commandValue) {
		cliCommands.put(commandName, commandValue);
		return;
	}
	
	/**
	 * Get a value, not a command value, from the main settings information.
	 * @param fieldName setting name
	 * @return value of setting
	 */
	public String get(String fieldName) {
		String sVal = mainSettings.get(fieldName);
		if (sVal == null) sVal = "";
		return sVal;
	}
	
	/**
	 * Set a particular field to the accompanying value.
	 * @param fieldName the field name to set
	 * @param fieldValue the field value
	 */
	public void set(String fieldName, String fieldValue) {
		mainSettings.put(fieldName, fieldValue);
		return;
	}
	
	/**
	 * Place an object in the config so it can be used by any class that needs it.  Sometimes
	 * one class is needed by many others, so this is an easy way to make it global.
	 * @param fieldName name object is identified by
	 * @param globalObject the class/object to store
	 */
	public void putObject(String fieldName, Object globalObject) {
		globalObjects.put(fieldName, globalObject);
		return;
	}
	
	/**
	 * Get an object that could be any type of class.
	 * @param fieldName identifier of object
	 * @return object
	 */
	public Object getObject(String fieldName) {
		Object globalObject = globalObjects.get(fieldName);
		return globalObject;
	}
	
	/**
	 * Find out if we have a particular object in the config.
	 * @param fieldName name of object
	 * @return true if we have that object
	 */
	public boolean hasObject(String fieldName) {
		boolean hasObject = false;
		if (globalObjects.containsKey(fieldName))
			hasObject = true;
		return hasObject;
	}
	
	/**
	 * Get a sorted listing of all the values stored in the config.
	 * @return array of all values, each line has the key, then tab, then the value
	 */
	public String[] listValues() {
		int x;
		String sKey, sVal;
		String[] allVals = new String[0];
		allVals = mainSettings.keySet();
		for (x = 0; x < allVals.length; x++) {
			sKey = allVals[x];
			sVal = mainSettings.get(sKey);
			allVals[x] = sKey + "\t" + sVal;
			System.out.println("Setting key: " + sKey + ", Val: " + sVal);
		}
		return allVals;
	}
	
	/**
	 * Get a sorted listing of all the commans stored in the config.
	 * @return array of all commands, each line has the key, then tab, then the value
	 */
	public String[] listCommands() {
		int x;
		String sKey, sVal;
		String[] allVals = new String[0];
		allVals = cliCommands.keySet();
		for (x = 0; x < allVals.length; x++) {
			sKey = allVals[x];
			sVal = cliCommands.get(sKey);
			allVals[x] = sKey + "\t" + sVal;
			System.out.println("Command key: " + sKey + ", Val: " + sVal);
		}
		return allVals;
	}
	
	/**
	 * Get a sorted listing oof all the objects stored here.
	 * @return array of all objects
	 */
	public String[] listObjects() {
		int iCount = 0;
		String[] allObjs;
		Set allKeys = globalObjects.keySet();
		Iterator iPoint = allKeys.iterator();
		allObjs = new String[allKeys.size()];
		while (iPoint.hasNext()) {
			allObjs[iCount++] = (String) iPoint.next();
			System.out.println("Object key: " + allObjs[iCount - 1]);
		}
		Arrays.sort(allObjs);
		return allObjs;
	}
	
	/**
	 * There are times status reports are printed to the command line and
	 * times they're printed to a text window.  Use this for outputs and the
	 * destination of the text is automatically determined according to other 
	 * program settings.  This does NOT print a CR at the end of the line.
	 * @param outputText text to output
	 */
	public void print(String outputText) {
		
		return;
	}
	
	/**
	 * There are times status reports are printed to the command line and
	 * times they're printed to a text window.  Use this for outputs and the
	 * destination of the text is automatically determined according to other 
	 * program settings.  This DOES print a CR at the end of the line.
	 * @param outputText text to output
	*/
	public void println(String outputText) {
		
		return;
	}

}
