package com.hal.gui;
/**
Program: TNet Client Letter Printer
Classes: helpWindow
Author : Hal Vaughan
Company: Threshold Digital, Ltd.
			4221 Wakefield Rd.
			Richmond, VA 23235
			(804)-560-2820
			thresholddigital.com
			hal@thresholddigital
Purpose: This is part of the set of classes that goes on a client system.
TNet is part of a system to prepare data for clients, which is then
custom printed in the client's office in formats they specify.  This
system retreives data via e-mail, decrypts it and verifies the signature,
then unzips archive files and prints out all files sent.  (Batch files
can also be sent to change settings.)  When done, we send e-mail back to
report to the server.  All printing and formatting is done with OpenOffice.org,
which is open source and available at openoffice.org.  We use this program
because it is free (as in speech and as in beer) and the data file formats are
clearly documented for us to use and modify (unlike certain other office suites).

Specifics:
	helpWindow: Creates a JPanel with basic directions and a button to press for help.
	Call with an array of strings for labels (1 per line), frame title, button text,
		window title, and help file
*/

import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JFrame;

public class HelpWindow extends JFrame implements ActionListener {

	private static final long serialVersionUID = 1L;
	
	HTMLPanel htmlPanel;
	
	JButton bCancel = null;

	/**
	 * Create a help window and specify the size.
	 * @param iWidth width of window
	 * @param iHeight height of window
	 * @param displayText text to display
	 * @param windowTitle title for window and etched border.
	 */
	public HelpWindow(int iWidth, int iHeight, String displayText, String windowTitle) {
		makeWindow(iWidth, iHeight, displayText, windowTitle);
	}

	/**
	 * Make a help window with the default size.
	 * @param displayText text to display
	 * @param windowTitle title for window and etched border
	 */
	public HelpWindow(String displayText, String windowTitle) {
		makeWindow(0, 0, displayText, windowTitle);
	}
	
	/**
	 * Do internal setup for constructors.  This is the stuff both have in common.
	 * @param iWidth width of window, if 0, we use default
	 * @param iHeight height of window, if 0 we use default
	 * @param displayText text to display in the HTML panel
	 * @param windowTitle title for both etched border and window title
	 */
	private void makeWindow(int iWidth, int iHeight, String displayText, String windowTitle) {
		Container cPane = this.getContentPane();
		htmlPanel = new HTMLPanel(displayText, windowTitle);	
		this.setTitle(windowTitle);
		bCancel = new JButton("Close");
		bCancel.addActionListener(this);
		cPane.setLayout(new BoxLayout(cPane, BoxLayout.Y_AXIS));
		bCancel.setAlignmentX(Component.CENTER_ALIGNMENT);
		cPane.add(htmlPanel);
		cPane.add(bCancel);
		if (iWidth == 0)
			iWidth = 400;
		if (iHeight == 0)
			iHeight = 250;
		this.setSize(new Dimension(iWidth, iHeight));
		GUIUtils.centerWindow(this);
		return;
	}
	
	/**
	 * Set the title of the window and the etched border.
	 */
	public void setTitle(String newTitle) {
		super.setTitle(newTitle);
		htmlPanel.setTitle(newTitle);
		return;
	}
	
	/**
	 * Set the text to display on the HTML panel (this means it WILL NOT be read as HTML!).
	 * @param displayText text to display
	 */
	public void setText(String displayText) {
		htmlPanel.setText(displayText);
		return;
	}
	
	/**
	 * Specify to use a specific text/html file as a source for the text
	 * for the HTML panel.
	 * @param sourceFile full file path to source file
	 */
	public void setFile(String sourceFile) {
		htmlPanel.setFile(sourceFile);
		return;
	}
	
	/**
	 * Determine whether or not to set an etched border around the HTML panel.
	 * @param useEtched true to use etched border
	 */
	public void useEtchedBorder(boolean useEtched) {
		htmlPanel.useEtchedBorder(useEtched);
		return;
	}
	
	/**
	 * Center this window over another.
	 * @param parentWindow window to center ourself over.
	 */
	public void centerOnWindow(JFrame parentWindow) {
		GUIUtils.centerWindowOnWindow(parentWindow, this);
		return;
	}	
	
	/**
	 * Listen to close window when asked to.
	 */
	public void actionPerformed(ActionEvent aEvent) {
		Object sourceControl = aEvent.getSource();
		if (sourceControl == bCancel) {
			this.setVisible(false);
		}
		return;
	}
}