package com.hal.oldtimeradio;

import java.io.File;
import java.util.Arrays;
import java.util.LinkedList;

import com.hal.conf.GeneralConfig;
import com.hal.util.HalFile;
import com.hal.util.StringHashMap;

public class EpisodeData {
	
	int iOrder = 1;
	
	String downDirectory, showTitle, epFile, downURL;

	public LinkedList allFiles;
	
	private StringHashMap myTitle, myShow, myFormat, myLicense, myFile, myOrder;
	
	GeneralConfig myConfig;
	
	ShowData allShows;

	/**
	 * Contstructor (D'oh!)
	 * @param programConfig the standard configuration class
	 * @param showData the info or data for all the OTR shows
	 */
	public EpisodeData(GeneralConfig programConfig, ShowData showData) {
		myConfig = programConfig;
		allShows = showData;
		downURL = myConfig.get("DownloadURL");
		downDirectory = myConfig.get("DownloadDirectory");
		reset();
	}
	
	/**
	 * Reset everything so we can start over again and create a new
	 * list of episodes.
	 */
	public void reset() {
		showTitle = "";
		epFile = "";
		allFiles = new LinkedList();
		myTitle = new StringHashMap();
		myShow = new StringHashMap();
		myFormat = new StringHashMap();
		myLicense = new StringHashMap();
		myFile = new StringHashMap();
		myOrder = new StringHashMap();
		iOrder = 1;
		return;
	}
	
	/**
	 * Load in a set of episodes from a particular show
	 * @param showTitle title of show that we load episodes for
	 */
	public void loadEpisodes(String showTitle) {
		int x;
		String sData;
		String[] aData;
		setNewShow(showTitle);
		downDirectory = myConfig.get("DownloadDirectory");		//Update in case it's been changed!
//		System.out.println("Loading episodes for show: " + showTitle + ", File: " + epFile);
		if (epFile == null || epFile == "") {
			return;
		}
		File fEpisodes = new File(epFile);
		reset();
		if (fEpisodes.exists()) {
			setNewShow(showTitle);
			sData = HalFile.fileToString(epFile);
			aData = sData.split("\n");
			for (x = 0; x < aData.length; x++) {
				addEpisodeLine(aData[x]);
			}
		} else {
			setNewShow(showTitle);
		}
		return;
	}
	
	/**
	 * Reset and specify a new show title.
	 * @param showTitle
	 */
	public void setNewShow(String showTitle) {
		reset();
		this.showTitle = showTitle;
		epFile = allShows.getEpisodeFile(showTitle);
//		System.out.println("New episode show: " + showTitle + ", Episode file: " + epFile);
		return;
	}
	
	/**
	 * Write out all the episodes to the episode file for the current show.
	 */
	public void saveEpisodes() {
		int x;
		String sData = "";
		String[] aEps = listSortedFiles();
		for (x = 0; x < aEps.length; x++) {
			sData = sData + getEpisodeLine(aEps[x]);
			if (x < aEps.length - 1) {
				sData = sData + "\n";
			}
		}
		HalFile.stringToFile(epFile, sData);
		return;
	}
	
	/**
	 * Add a new episode, not from a file, but a new downloaded episode.
	 * @param fileName file name (taken from URL)
	 * @param showTitle title of the show this episode is from
	 * @param episodeFormat audio format
	 * @param episodeLicense any licensing information
	 * @return the episode title
	 */
	public String newEpisode(String fileName, String showTitle, String episodeFormat, String episodeLicense) {
		String episodeTitle = makeTitle(fileName);
		addFile(fileName);
		myTitle.put(fileName, episodeTitle);
		myShow.put(fileName, showTitle);
		myFormat.put(fileName, episodeFormat);
		myLicense.put(fileName, episodeLicense);
		myOrder.put(fileName, String.valueOf(iOrder++));
		return episodeTitle;
	}
	
	/**
	 * Add an episode by adding a line from a text file.
	 * @param titleLine Line from text file with all fields separated by commas
	 * @return the episode file name
	 */
	public String addEpisodeLine(String titleLine) {
		int x, iLimit;
		String thisFile = "";
		String[] origParts = titleLine.split("\t");
		String[] allParts = new String[] {"", "", "", "", "", ""};
		iLimit = allParts.length;
		if (origParts.length < iLimit) {iLimit = origParts.length;}
		for (x = 0; x < iLimit; x++) {
			allParts[x] = origParts[x];
		}
		addEpisode(allParts[0], allParts[1], allParts[2], allParts[3], allParts[4], allParts[5]);
		thisFile = allParts[1];
		return thisFile;
	}
	
	/**
	 * Add an episode to the list if we've got all the data items for it
	 * @param episodeTitle title of the episode
	 * @param fileName episode file
	 * @param showTitle title of the show it belongs to
	 * @param episodeFormat audio format
	 * @param episodeLicense licensing info
	 */
	public void addEpisode(String episodeTitle, String fileName, String showTitle, String episodeFormat, String episodeOrder, String episodeLicense) {
		addFile(fileName);
		myTitle.put(fileName, episodeTitle);
		myShow.put(fileName, showTitle);
		myFormat.put(fileName, episodeFormat);
		myOrder.put(fileName, episodeOrder);
		myLicense.put(fileName, episodeLicense);
		return;
	}
	
	/**
	 * Add a file to the master file list
	 * @param newFile file name
	 */
	private void addFile(String newFile) {
		allFiles.add(newFile);
		String filePath = downDirectory + File.separator + allShows.getDirectory(showTitle) + File.separator + newFile;
//		System.out.println("Adding episode, File: " + newFile + ", Local file: " + filePath);
		myFile.put(newFile, filePath);
		return;
	}

	/**
	 * Create an episode line to be used for writing out a text file
	 * @param fileName name of the file we're working with
	 * @return tab separated line with episode data
	 */
	public String getEpisodeLine(String fileName) {
		String episodeLine = "";
		episodeLine = myTitle.get(fileName) + "\t" + fileName + "\t" + myShow.get(fileName) + 
				"\t" + myFormat.get(fileName) + "\t" + myOrder.get(fileName) + "\t" + myLicense.get(fileName);
		return episodeLine;
	}
	
	/**
	 * Make a title from a file name.  Eventually this will be adapted with the title editor.
	 * @param fileName episode file name
	 * @return title we get from file name
	 */
	public String makeTitle(String fileName) {
		int iPoint;
		String episodeTitle = fileName;
		iPoint = episodeTitle.lastIndexOf(".");
		if (iPoint >= 0) {
			episodeTitle = episodeTitle.substring(0, iPoint);
		}
//		System.out.println("First title filter: " + episodeTitle);
		episodeTitle = fileName.replace("_", " ");
//		for (x = 0; x < allChars.length; x++) {
//			String thisChar = allChars[x];
//			if (!thisChar.equals(thisChar.toLowerCase()) && x > 0) {
//				episodeTitle = episodeTitle + " " + thisChar;
//			} else if (thisChar.equals("_")){
//				episodeTitle = episodeTitle + " ";
//			} else {
//				episodeTitle = episodeTitle + thisChar;
//			}
//		}
		return episodeTitle;
	}

	/**
	 * Return an alphabetized list of the episode files.  This often will put the episodes in
	 * origina air order since many uploaders put the original air date at the start of the title. 
	 * @return sorted files
	 */
	public String[] listSortedFiles() {
		String[] aFiles= new String[allFiles.size()];
		aFiles = (String[]) allFiles.toArray(aFiles);
		Arrays.sort(aFiles);
		return aFiles;
	}
	
	/**
	 * List all the files, but not in any particular order.
	 * @return the list of sorted episode files
	 */
	public String[] listFiles() {
		String[] aFiles = new String[allFiles.size()];
		aFiles = (String[]) allFiles.toArray(aFiles);
		return aFiles;
 	}
	
	/**
	 * List all the titles of the episodes
	 * @return a list of episode titles
	 */
	public String[] listTitles() {
		int x;
		String[] allFiles, allTitles;
		allFiles = listFiles();
		allTitles = new String[allFiles.length];
		for (x = 0; x < allFiles.length; x++) {
			allTitles[x] = myTitle.get(allFiles[x]);
		}
		return allTitles;
		
	}
	
	/**
	 * List the episode titles for a string of episodes.
	 * @param allFiles files to get the titles for
	 * @return all the titles for the specified episodes
	 */
	public String[] listTitles(String[] allFiles) {
		int x;
		String[] allTitles;
		allTitles = new String[allFiles.length];
		for (x = 0; x < allFiles.length; x++) {
			allTitles[x] = myTitle.get(allFiles[x]);
		}
		return allTitles;
		
	}

	/**
	 * get the show title that goes with this particular episode.
	 * @param fileName name of file
	 * @return the title of the show this episode is part of
	 */
	public String getTitle(String fileName) {
		String sValue = myTitle.get(fileName);
		return sValue;
	}
	
	/**
	 * Get the show that this file is an episode of
	 * @param fileName file name
	 * @return the name of the show or series this is an episode of
	 */
	public String getShow(String fileName) {
		String sValue = myShow.get(fileName);
		return sValue;
	}
	
	/**
	 * Get the audio format of an episode file.
	 * @param fileName episode file name
	 * @return audio format
	 */
	public String getFormat(String fileName) {
		String sValue = myFormat.get(fileName);
		return sValue;
	}
	
	/**
	 * Get the licensing information to go with the file.
	 * @param fileName name of file we want licensing info for
	 * @return whatever text was supplied as the license by archive.org
	 */
	public String getLicense(String fileName) {
		String sValue = myLicense.get(fileName);
		return sValue;
	}
	
	/**
	 * Get the directory where the file will be stored.  This is the directory associated with the show
	 * this file is an episode of
	 * @param fileName the name of the episode file
	 * @return the show directory
	 */
	public String getDirectory(String fileName) {
		String sValue = allShows.getDirectory(getShow(fileName));
		return sValue;
	}
	
	/**
	 * Get the URL for the given file name
	 * @param fileName name of file
	 * @return URL for downloading the file
	 */
	public String getURL(String fileName) {
		String sValue = downURL + "/" + getDirectory(fileName) + "/" + fileName;
		return sValue;
	}
	
	/**
	 * Get the full path of where the file is stored (or will be stored) locally.
	 * @param fileName show file name
	 * @return full path to file
	 */
	public String getFilePath(String fileName) {
		String filePath = myFile.get(fileName);
		return filePath;
	}
	
	/**
	 * Get the sequence of the file according to the original archive.org xml file listing.
	 * @param fileName file name
	 * @return the position in sequence of the file
	 */
	public String getFileOrder(String fileName) {
		String sPoint = myOrder.get(fileName);
		return sPoint;
	}
	
	/**
	 * Check to see if a file exists localy, which tells us it has been downloaded.
	 * @param fileName name of file
	 * @return true if we have it on the disc.
	 */
	public boolean isDownloaded(String fileName) {
		boolean isDownloaded;
//		System.out.println("Filename: " + fileName + ", Local file: " + myFile.get(fileName));
		File fFile = new File(myFile.get(fileName));
		isDownloaded = fFile.exists();
		return isDownloaded;
	}
}
