package com.hal.oldtimeradio;

import java.io.File;
import java.util.LinkedList;

import com.hal.util.HalFile;
import com.hal.util.HalUtil;
import com.hal.util.StringHashMap;

public class TitleFilter {
	
	public boolean fixUnderline = true, fixSpaceBeforeCaps = true, fixExtension = true, fixWhitespace = true;
	boolean[] getTextFlags;
	String filePath = "";
	String[] filterDesc;
	public String defaultFilterFile = "otrDefault";  //  @jve:decl-index=0:
	
	LinkedList allRules;
	
	StringHashMap ruleIdx = new StringHashMap();
	
	/**
	 * Simple constructor.  Don't offer a non-filename version because while it might be
	 * needed, it's not a good idea.
	 * @param filePath path of of title filter file to load.  If null, we don't load
	 */
	public TitleFilter(String filePath) {
		this.filePath = filePath;
		setRules();
		if (!filePath.equals(""))
			loadFilter(filePath);
	}
	
	/**
	 * Set the filter descriptions.
	 */
	private void setRules() {
		int x;
		filterDesc = new String[] {
				"Remove all non-letters from the start", 
				"Remove up to the first non-letter",
				"Remove up through a phrase", 
				"Remove up to a phrase", 
				"Remove up through the first date",				
				"Remove up to the first date",

				"Remove all non-letters from the end",
				"Remove everything after the last non-letter",
				"Remove from a phrase on to the end", 
				"Remove everything after a phrase", 
				"Remove from the last date to the end",
				"Remove everything after the last date"
		};
		getTextFlags = new boolean[] {
				false,
				false,
				true,
				true,
				false,
				false,
				
				false,
				false,
				true,
				true,
				false,
				false
		};
		for (x = 0; x < filterDesc.length; x++) {
			ruleIdx.put(filterDesc[x], x);
		}
		return;
	}
	
	/**
	 * Get a list of all the standard filter rule descriptions.  Used
	 * when listing choices for the user.
	 * @return array of all filter rules
	 */
	public String[] getRuleDescriptions() {
		return filterDesc;
	}
	
	/**
	 * Load all the info a filter needs.  If the file does not exist,
	 * then load the default filter for it.  If the default file does not
	 * exist, then create the settings.
	 * @param filePath path of file for filter info
	 */
	public void loadFilter(String filePath) {
		String fileParent, tempPath;
		this.filePath = filePath;
		File oFile = new File(filePath);
		if (oFile.exists()) {
			readFile(filePath);
		} else {
			fileParent = oFile.getParent();
			tempPath = fileParent + File.separator + defaultFilterFile + ".ttl";
			oFile = new File(tempPath);
			if (oFile.exists()) {
				readFile(tempPath);
			} else {
				allRules = new LinkedList();
				fixUnderline = true;
				fixSpaceBeforeCaps = true;
				fixExtension = true;
				fixWhitespace = true;
			}
		}
		return;
	}
	
	/**
	 * Read the actual filter file and set the boolean variables, then put all
	 * the filter rules in a linked list.
	 * @param filePath file to read in
	 */
	private void readFile(String filePath) {
		int x, iPoint, iSeq;
		String sLine, sSeq, sVal;
		String sData = HalFile.fileToString(filePath);
		String[] aData = sData.split("\n");
		String[] aRules = new String[sData.length()];
		fixUnderline = false;
		fixSpaceBeforeCaps = false;
		fixExtension = false;
		fixWhitespace = false;
		allRules = new LinkedList();
		for (x = 0; x < aData.length; x++) {
			sLine = aData[x];
			sVal = "";
			iPoint = sLine.indexOf("=");
			if (iPoint > 0) {
				sVal = sLine.substring(iPoint + 1);
			}
			if (sLine == null || sLine.equals("") || sLine.startsWith("#")) continue;
			if (sLine.startsWith("rule")) {
				iPoint = sLine.indexOf("=");
				sSeq = sLine.substring(4, iPoint);
				iSeq = Integer.parseInt(sSeq);
				aRules[iSeq] = sVal;
			} else if (sLine.startsWith("fix")) {
				if (sLine.startsWith("fixunderline") && sVal.equals("true")) fixUnderline = true;
				if (sLine.startsWith("fixspacebeforecaps") && sVal.equals("true")) fixSpaceBeforeCaps = true;
				if (sLine.startsWith("fixextension") && sVal.equals("true")) fixExtension = true;
				if (sLine.startsWith("fixwhitespace") && sVal.equals("true")) fixWhitespace = true;
			}
		}
		for (x = 0; x < aRules.length; x++) {
			if (aRules[x] == null) continue;
			allRules.add(aRules[x]);
		}
		return;
	}
	
	/**
	 * Save the filter to a specific file.  Once the file name is
	 * changed, it stays changed so we will be working with a different file.
	 * @param filePath file to save to
	 */
	public void saveFilter(String filePath) {
		this.filePath = filePath;
		saveFilter();
		return;
	}
	
	/**
	 * Save the filter info.  No file name is specified because we save
	 * to the file we loaded from.  If we save differently, we use the
	 * other, overloaded, version of this method.
	 */
	public void saveFilter() {
		int x;
		String sOut = "";
		sOut = "#Simple filter rules:\n";
		sOut = sOut + "fixunderline=" + fixUnderline + "\n";
		sOut = sOut + "fixspacebeforecaps=" + fixSpaceBeforeCaps + "\n";
		sOut = sOut + "fixextension=" + fixExtension + "\n";
		sOut = sOut + "fixwhitespace=" + fixWhitespace + "\n";
		sOut = sOut + "\n#Filter processing rules:\n";
		for (x = 0; x < allRules.size(); x++) {
			sOut = sOut + "rule" + x + "=" + (String) allRules.get(x) + "\n";
		}
//		System.out.println("Saving fitler to file: " + filePath);
		HalFile.stringToFile(filePath, sOut);
		return;
	}

	/**
	 * Move a rule up in the processing list.  By up, that means farther up in
	 * the order if viewed in a list box, as in the index decreases by one.
	 * @param ruleIdx pointer to the rule to move.
	 */
	public void moveRuleUp(int ruleIdx) {
		if (ruleIdx >= 1) {
			swapRules(ruleIdx, ruleIdx - 1);
		}
		return;
	}
	
	/**
	 * Move a rule down in the processing list.  By down, that means farther down in
	 * the order if viewed in a list box, as in the index increases by one.
	 * @param ruleIdx pointer to the rule to move.
	 */
	public void moveRuleDown(int ruleIdx) {
		if (ruleIdx < allRules.size() - 1) {
			swapRules(ruleIdx, ruleIdx + 1);
		}
		return;
	}
	
	/**
	 * Swap the order of two filter processing rules
	 * @param rule1 the pointer to the first rule
	 * @param rule2 the pointer to the second rule
	 */
	public void swapRules(int rule1, int rule2) {
		String swapRule = (String) allRules.get(rule1);
		allRules.set(rule1, allRules.get(rule2));
		allRules.set(rule2, swapRule);
		return;
	}
	
	/**
	 * Add a rule to the current list of rules
	 * @param ruleText rule to add
	 */
	public void addRule(String ruleText) {
		allRules.add(ruleText);
		return;
	}
	
	/**
	 * Delete a rule from the list
	 * @param ruleIdx the index of the rule to delete
	 */
	public void deleteRule(int ruleIdx) {
		allRules.remove(ruleIdx);
		return;
	}
	
	/**
	 * Check to see if a rule needs extra text or not
	 * @param ruleText rule text (anything after a colon is stripped)
	 * @return true if it needs text input
	 */
	public boolean ruleNeedsText(String ruleText) {
		boolean needsText = false;
		int iPoint;
		if (ruleText.contains(":")) {
			iPoint = ruleText.indexOf(":");
			ruleText = ruleText.substring(0, iPoint);
		}
		iPoint = Integer.parseInt(ruleIdx.get(ruleText));
		if (getTextFlags[iPoint])
			needsText = true;
		return needsText;
	}
	
	/**
	 * Get a list of all the rules in this filter
	 * @return list of rules
	 */
	public String[] getRules() {
		String[] ruleList = new String[allRules.size()]; 
		ruleList = (String[]) allRules.toArray(ruleList);
		return ruleList;
	}

	/**
	 * Get the size of the rule set.
	 * @return the size of the rule set
	 */
	public int getRuleSize() {
		return allRules.size();
	}
	
	/**
	 * Do the actual filtering of a title.  Process a file name given to us by all
	 * the rules set for the specified show.
	 * @param episodeTitle title of the episode to filter
	 * @return filtered and fixed title name
	 */
	public String filterTitle(String episodeTitle) {
		int x, iPoint, iTail;
		String newTitle = episodeTitle, sRule, sText;
		iPoint = newTitle.lastIndexOf(".");
		if (fixExtension && iPoint > -1) {
			newTitle = newTitle.substring(0, iPoint);
		}
		for (x = 0; x < allRules.size(); x++) {
			if (fixWhitespace) newTitle = newTitle.trim();
			sRule = (String) allRules.get(x);
			sText = "";
			iPoint = sRule.indexOf(":");
			if (iPoint > 0) {
				iTail = iPoint;
				if (iTail < sRule.length() - 1)
					iTail++;
				sText = sRule.substring(iTail);
				sRule = sRule.substring(0, iPoint);
			}
//			0	"Remove all non-letters from the start",
			if (sRule.equals(filterDesc[0]))
				newTitle = HalUtil.doRegex(newTitle, "^[^a-zA-Z]*", "");
//			1	"Remove up to the first non-letter",
			if (sRule.equals(filterDesc[1]))
				newTitle = HalUtil.doRegex(newTitle, "^.*?([^a-zA-Z])", "$1");
//			2	"Remove up through a phrase",
			if (sRule.equals(filterDesc[2]))
				newTitle = HalUtil.doRegex(newTitle, "^.*?" + sText, "");
//			3	"Remove up to a phrase",
			if (sRule.equals(filterDesc[3]))
				newTitle = HalUtil.doRegex(newTitle, "^.*?(" + sText + ")", "$1");
//			4	"Remove up through the first date",
			if (sRule.equals(filterDesc[4]))
				newTitle = HalUtil.doRegex(newTitle, "^.*?[0-9]{1,4}(-|_)[0-9]{1,4}(-|_)[0-9]{1,4}", "");	
//			5	"Remove up to the first date",
			if (sRule.equals(filterDesc[5]))
				newTitle = HalUtil.doRegex(newTitle, "^.*?([0-9]{1,4}(-|_)[0-9]{1,4}(-|_)[0-9]{1,4})", "$1");

//			6	"Remove all non-letters from the end",
			if (sRule.equals(filterDesc[6]))
				newTitle = HalUtil.doRegex(newTitle, "[^a-zA-Z]*$", "");
//			7	"Remove everything after the last non-letter",
			if (sRule.equals(filterDesc[7]))
				newTitle = HalUtil.doRegex(newTitle, "^(.*[^a-zA-Z])[a-zA-Z].*?$", "$1");
//			8	"Remove from a phrase on to the end",
			if (sRule.equals(filterDesc[8]))
				newTitle = HalUtil.doRegex(newTitle, "^(.*)" + sText + ".*?$", "$1");
//			9	"Remove everything after a phrase", 
			if (sRule.equals(filterDesc[9]))
				newTitle = HalUtil.doRegex(newTitle, "^(.*" + sText + ").*?$", "$1");
//			10	"Remove from the last date to the end",
			if (sRule.equals(filterDesc[10]))
				newTitle = HalUtil.doRegex(newTitle, "^(.*)[0-9]{1,4}(-|_)[0-9]{1,4}(-|_)[0-9]{1,4}.*?$", "$1");
//			11	"Remove everything after the last date"
			if (sRule.equals(filterDesc[11]))
				newTitle = HalUtil.doRegex(newTitle, "^(.*[0-9]{1,4}(-|_)[0-9]{1,4}(-|_)[0-9]{1,4}).*?$", "$1");
			
		}
		if (fixUnderline) {
			newTitle = newTitle.replaceAll("_", " ");
//			newTitle = HalUtil.doRegex(newTitle, "_", " ");	
		}
		if (fixWhitespace) {
			newTitle = newTitle.trim();
			newTitle = HalUtil.doRegex(newTitle, " +", " ");
		}
		if (fixSpaceBeforeCaps) {
			newTitle = newTitle.replaceAll("([a-z0-9])([A-Z])", "$1 $2");
			newTitle = newTitle.replaceAll("([a-zA-Z])([0-9])", "$1 $2");
//			newTitle = HalUtil.doRegex(newTitle, "([a-z])([A-Z])", "$1 $2");
		}
//		newTitle = "Filtered: " + episodeTitle;
		return newTitle;
	}
}
