package com.hal.util;
/**
Class  : TNFile
Author : Hal Vaughan
Company: Threshold Digital, Ltd.
			4221 Wakefield Rd.
			Richmond, VA 23235
			(804)-560-2820
			thresholddigital.com
			hal@thresholddigital

	This is not a class with any objects.  It is only a utility class to
	provide a lot of file handling methods:

	fileToString: read a file into a string

	fileToBytes: read in a file to a byte[] array

	stringToFile: save a string as a file

	bytesToFile: save a byte[] array as a file

	makeName: create a name in a directory when given a suffix.  We use the date & time & a 4 digit
	counter to create a unique filename.

	listDirLong: list the directory, with full path names

	listDirShort: list the directory, with short file names
*/

import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.net.URL;
import java.util.Arrays;
import java.util.Calendar;

/**
 * Handle all the tedious file work for the threshNet Agent. 
 * @author Hal
 *
 */
public class HalFile {

	public boolean isJar = false;
	
	private static String sErrorFNF = "Error: File not found";
	private static String sErrorIOE = "Error: IO Exception";
	private static byte[] bErrorFNF = sErrorFNF.getBytes();
	private static byte[] bErrorIOE = sErrorIOE.getBytes();
	
	private Writer fileWriter = null;
	
	private FileOutputStream byteFileOut = null;
	private BufferedOutputStream bufferedOut = null;

	/**
	 * Load a file and convert it to a string
	 * @param fileName file to load
	 * @return file contents as a string or empty string if nothing read in
	 */
	public static String fileToString (String fileName) {
		boolean errorFlag = false;
		int iPoint = 0, inRead = -2, iLimit;
		char[] inBuff;
        BufferedReader inStream = null;
        String inData = null;
		File oFile = new File(fileName);
        iLimit = (int) oFile.length();
        inBuff = new char[iLimit];
        try {
        	inStream = new BufferedReader(new FileReader(fileName));
			while (inRead != -1 && iPoint < iLimit) {
				inRead = inStream.read(inBuff, iPoint, iLimit);
				iPoint = iPoint + inRead;
				iLimit = iLimit - inRead;
			}
        } catch (Exception e) {
        	errorFlag = true;
        	System.out.println("Errors reading file: " + fileName);
        	e.printStackTrace();
        	try {
        		inStream.close();
        	} catch (Exception ex) {
        		System.out.println("Cannot close file after other errors: " + fileName);
        		ex.printStackTrace();
        	}
        }
    	try {
    		inStream.close();
    	} catch (Exception ex) {
    		System.out.println("Cannot close file: " + fileName);
    		ex.printStackTrace();
    	}
        if (errorFlag)
        	inData = "";
        else
        	inData =  new String(inBuff);
		return inData;
	}

	/**
	 * Load a file and return the contents as a byte[]
	 * @param sName file to load
	 * @return contents of file as byte[]
	 */
	public static byte[] fileToBytes (String sName) {
		byte[] bNull = new byte[0];
		int iLen = 0;
//		System.out.println("DEBUG: Reading file: " + sName);
		File fInfo = new File(sName);
		if (!fInfo.exists()) {return bNull;}
		FileInputStream fis;
		iLen = (int) fInfo.length();
		byte[] bData = new byte[iLen];
		try {fis = new FileInputStream(sName);}
				catch (FileNotFoundException e) {return bErrorFNF;}
		DataInputStream dis = new DataInputStream(fis);
		try {
			dis.read(bData, 0, iLen);
			dis.close();
			fis.close();
		} catch (IOException e) {
			return bErrorIOE;
		}
		return bData;
	}

	/**
	 * Save a string to a file name
	 * @param sName name of file
	 * @param sData data to save
	 * @return true if save was successful
	 */
	public static String stringToFile (String sName, String sData) {
		String sState = "";
		sState = bytesToFile(sName, sData.getBytes(), false);
		return sState;
	}
	
	/**
	 * Write an array to a file.  Do it line by line, no concat work due to
	 * possible memory issues.
	 * @param fileName name of file to write to
	 * @param aData data to write out to file
	 * @return blank unless an exception occured.
	 */
	public static String arrayToFile(String fileName, String[] aData) {
		int x;
		String sError = "";
		File fOut = new File(fileName);
		Writer outStream = null; 
		FileWriter fWrite = null;
		try {
			fWrite = new FileWriter(fOut);
			outStream = new BufferedWriter(fWrite);
			for (x = 0; x < aData.length; x++) {
				outStream.write(aData[x]);
			}
			outStream.close();
		} catch (Exception e) {sError = e.toString();}
		return sError;
	}
	
	/**
	 * Easy collection to write a text file out without doing the streams.  Open here, then write,
	 * then close, all error trapphing annd such is done here and we return any errors
	 * @param fileName name of file to open for writing to
	 * @return the text of any error
	 */
	public String openTextFileOut(String fileName) {
		String sError = "";
		File fOut = new File(fileName);
		FileWriter fWrite = null;
		try {
			fWrite = new FileWriter(fOut);
			fileWriter = new BufferedWriter(fWrite);
		} catch (Exception e) {sError = e.toString();}
		return sError;
	}

	
	/**
	 * Write more text data to the open file.  Errors returned as text
	 * @param outData String to write to file
	 * @return any error (or empty if no error)
	 */
	public String writeTextOut(String outData) {
		String sError = "";
		try {
			fileWriter.write(outData);
		} catch (Exception e) {sError = e.toString();}
		return sError;
	}
	
	/**
	 * Close the output file.
	 * @return any error codes
	 */
	public String closeTextFileOut() {
		String sError = "";
		try {
			fileWriter.close();
		} catch (Exception e) {sError = e.toString();}
		return sError;
	}
	
	/**
	 * Easy collection to write a binary (or any) file out without doing the streams.  Open here, then write,
	 * then close, all error trapphing annd such is done here and we return any errors
	 * @param fileName name of file to open for writing to
	 * @return the text of any error
	 */
	public String openByteFileOut(String fileName) {
		String sError = "";
		try {
			byteFileOut = new FileOutputStream(fileName);
			bufferedOut = new BufferedOutputStream(byteFileOut);
		} catch (Exception e) {
			sError = e.toString();
		}
		return sError;
	}
	
	/**
	 * Write more binary data to the open file.  Errors returned as text
	 * @param outBytes byte[] to write to file
	 * @return any error (or empty if no error)
	 */
	public String writeBytesOut(byte[] outBytes) {
		String sError = "";
		try {
			bufferedOut.write(outBytes);
		} catch (Exception e) {
			sError = e.toString();
		}
		System.out.println("Writing byte to file, bytes: " + outBytes.length + ", Error codes: " + sError);
		return sError;
	}
	
	/**
	 * Write out text data to the binary file we're writing to.
	 * @param outString text to write to file
	 * @return any error codes in text form
	 */
	public String writeBytesOut(String outString) {
		String sError = "";
		byte[] outBytes = outString.getBytes();
		try {
			byteFileOut.write(outBytes);
		} catch (Exception e) {
			sError = e.toString();
		}
		return sError;
	}
	
	/**
	 * Close the output file.
	 * @return any error codes
	 */
	public String closeByteFileOut() {
		String sError = "";
		try {
			bufferedOut.close();
		} catch (Exception e) {
			sError = e.toString();
		}
		
		return sError;
	}

	/**
	 * Append a string to an already existing file. 
	 * @param sName file to append to
	 * @param sData string to append to file
	 * @return true if successful
	 */
	public static String appendStringToFile (String sName, String sData) {
		String sState = "";
		sState = bytesToFile(sName, sData.getBytes(), true);
		return sState;
	}

	/**
	 * Save a byte[] to a file
	 * @param sName name of file to save to
	 * @param bData bytes to write to file
	 * @param bAppend true if we append to what's already there, false we make a new file
	 * @return true if write was successful
	 */
	private static String bytesToFile (String sName, byte[] bData, boolean bAppend) {
//		System.out.println("File: "+sName+", Length: "+bData.length);
		String sState = "true";
		FileOutputStream fOut = null;
		File fFile = new File(sName);
		try {fOut = new FileOutputStream(fFile, bAppend);}
				catch (FileNotFoundException e) {sState = sErrorFNF;}
		try {
			fOut.write(bData);
			fOut.close();
		} catch (IOException e) {
			sState = sErrorIOE;
			System.out.println("Error writing to file: " + sState);
		}
		return sState;
	}

	/**
	 * Overload, does not append but just saves byte[] to a file
	 * @param sName name of file
	 * @param bData data to write to file
	 * @return true if write was successful
	 */
	public static String bytesToFile(String sName, byte[] bData) {
		return bytesToFile(sName, bData, false);
	}

	/**
	 * Overloaded version, just append data to a file
	 * @param sName file to apend to
	 * @param bData data to append
	 * @return true if write was successful
	 */
	public static String appendBytesToFile(String sName, byte[] bData) {
		return bytesToFile(sName, bData, true);
	}

	/**
	 * Load a file and convert each line to a key/value pair (separated by the equals sign)
	 * and put it in a HashMap
	 * @param fName file to read
	 * @return HashMap of file data in key/value pairs (all as Strings)
	 */
	public static StringHashMap fileToHash (String fName) {
		StringHashMap hVals = new StringHashMap();
		String[] aLine;
		String sData = fileToString(fName);
		String[] aData = sData.split("\n");
// 		System.out.println("DEBUG: Hash count: " + aData.length);
		for (int i = 0; i < aData.length; i++) {
			if (aData[i].startsWith("#")) {continue;}
			if (aData[i].startsWith("//")) {continue;}
			if (aData[i].endsWith("=")) {
				aLine = new String[2];
				aLine[0] = aData[i].substring(0, aData[i].length() - 1);
				aLine[1] = "";
			} else {
				aLine = aData[i].split("=");
			}
			if (aLine.length < 2) {
				aLine = aData;
			}
			if (aLine[1].equals("null")) {aLine[1] = "";}
// 			System.out.println("DEBUG Hashing, Key: "+aLine[0].trim()+", Value: "+aLine[1].trim());
			hVals.put(aLine[0].trim(), aLine[1].trim());
		}
//		String[] xList = hVals.keySet();
		return hVals;
	}

	/**
	 * If we're in a jar, find out our full path, including the jar file.
	 * @return full path for this jar
	 */
	public String getOurPath(Object sourceObject) {
		int iPoint;
		String ourPath, thisName;
		Class thisClass;
		URL thisURL;

		thisClass = sourceObject.getClass();
		thisName = thisClass.getName();
		thisURL = thisClass.getResource("/" + thisName + ".class");
		if (thisURL == null) {
			thisName = thisName.replace(".", "/");
			thisURL = thisClass.getResource("/" + thisName + ".class");
		}
		ourPath = thisURL.toString();
		if (ourPath.contains(".jar!")) {
			isJar = true;
			ourPath = ourPath.replace("jar:file:", "");
			iPoint = ourPath.lastIndexOf(".jar!");
			ourPath = ourPath.substring(0, iPoint + 4);
		} else {
			isJar = false;
			ourPath = ourPath.replace("file:", "");
		}
		return ourPath;
	}
	
	/**
	 * If we're in a jar, get our parent directory, including the last file sep character.
	 * @return the parent to the jar we're in.
	 */
	public String getOurParent(Object sourceObject) {
		int iPoint;
		String jarPath, parentPath;
		jarPath = getOurPath(sourceObject);
		iPoint = jarPath.lastIndexOf(File.separator);
		parentPath = jarPath.substring(0, iPoint + 1);
		return parentPath;
	}
	
	/**
	 * Get just a plain directory listing of a specified directory (the full path of the directory).
	 * @param dName full path of directory to list
	 * @return list of that directory
	 */
	public static String[] listDirectory(String dName) {
		File sDir = new File(dName);
		String[] fList = sDir.list();
		if (fList == null) {fList = new String[0];}
		Arrays.sort(fList);
		return fList;
	}

	/**
	 * List files in a directory and return a sorted list of their full pathnames.
	 * @param dName directory to list
	 * @return list of files, with full path names
	 */
	public static String[] listDirectoryLong(String dName) {
		File oFile = new File(dName);
		File[] aFile = oFile.listFiles();
		if (aFile == null) {aFile = new File[0];}
		String[] sFile = new String[aFile.length];
		for (int i = 0; i < aFile.length; i++) {
			sFile[i] = aFile[i].getPath();
		}
		Arrays.sort(sFile);
		return sFile;
	}

	/**
	 * Get a recursive listing of a directory.
	 * @param sDir directory to list
	 * @return all files, recursively listed, each level indented an extra space
	 */
	public static String[] deepList(String sDir) {
		String[] allFiles = deepList(sDir, 0);
		return allFiles;
	}
	
	/**
	 * Get a deep recursive listing of a directory and also do the MD5 checksums on each
	 * file so we can use it for troubleshooting.
	 * @param sDir directory to list
	 * @param inTab how many levels deep (for recursion), start with 0 on actual first call
	 * @param iLen length of file names - names longer than this are truncated.
	 * @return list of all the files.
	 */
	private static String[] deepList(String sDir, int inTab) {
//		System.out.println("Listing directory.  Level: "+inTab+", Length: "+iLen+", Dir: "+sDir);
		int x;
		String sBlank = "";
		String[] fileList = null;
		String[] tList = null;
		String[] xList = null;
		if (inTab == 0) {fileList = new String[] {sDir};}
		File oFile = new File(sDir);
		sDir = oFile.getPath() + File.separator;
		for (x = 0; x < inTab; x++) {sBlank = sBlank + " ";}
		String[] sFile = oFile.list();
		for (x = 0; x < sFile.length; x++) {
			sFile[x] = sDir + sFile[x];
			oFile = new File(sFile[x]);
			if (oFile.isDirectory()) {
				tList = deepList(sFile[x], inTab + 1);
				if (fileList != null) {
					xList = (String[]) fileList.clone();
					fileList = new String[tList.length + xList.length];
					System.arraycopy(tList, 0, fileList, 0, tList.length);
					System.arraycopy(xList, 0, fileList, tList.length, xList.length);
				} else {fileList = (String[]) tList.clone();}
			}
			sFile[x] = sBlank + sFile[x];
		}
		if (fileList != null) {
			xList = (String[]) fileList.clone();
			fileList = new String[sFile.length + xList.length];
			System.arraycopy(sFile, 0, fileList, 0, sFile.length);
			System.arraycopy(xList, 0, fileList, sFile.length, xList.length);
		} else {fileList = (String[]) sFile.clone();}
		return fileList;
	}

	/**
	 * Many times we need sequential file names, such as when generating print files.  This
	 * will provide a unique name and make sure the name is stored as the next file in the
	 * directory when alpha sorted.  The format will be YYYY-MMDD-HHMMSS=xxxx, where xxxx
	 * is a unique number.
	 * @param sDir directory file will be in
	 * @param sSuffix suffix for filename
	 * @return unique full path of file name
	 */
	public static String makeName(String sDir, String sSuffix) {
		boolean bDone = false;
		int iCount = 0;
		String sCount = "", sName = "", sYear = "", sMonth = "", sDate = "",
			sHour = "", sMin = "", sSeconds = "";
		File oFile;
		Calendar oDate = Calendar.getInstance();
		oFile = new File(sDir);
		sDir = oFile.getPath() + File.separator;
		while (!bDone) {
			sMonth = "00" + String.valueOf(oDate.get(Calendar.MONTH) + 1);
			sDate = "00" + String.valueOf(oDate.get(Calendar.DATE));
			sHour = "00" + String.valueOf(oDate.get(Calendar.HOUR_OF_DAY));
			sMin = "00" + String.valueOf(oDate.get(Calendar.MINUTE));
			sSeconds = "00" + String.valueOf(oDate.get(Calendar.SECOND));
			sMonth = sMonth.substring(sMonth.length() - 2);
			sDate = sDate.substring(sDate.length() - 2);
			sHour = sHour.substring(sHour.length() - 2);
			sMin = sMin.substring(sMin.length() - 2);
			sSeconds = sSeconds.substring(sSeconds.length() - 2);
			sYear = String.valueOf(oDate.get(Calendar.YEAR)) + "-" + sMonth + sDate + "-" +
					sHour + sMin + sSeconds + "-";
			iCount = 0;
			while (!bDone) {
				sCount = "0000" + String.valueOf(iCount);
				sCount = sCount.substring(sCount.length() - 4, sCount.length());
				sName = sDir + sYear + sCount + "-" + sSuffix;
				oFile = new File(sName);
				if (!oFile.exists()) {bDone = true; break;}
				iCount++;
				if (iCount >= 10000) {break;}
			}
		}
		return sName;
	}

	/**
	 * Copy a file to another directory.  We specify the full path name and the directory to move it to.
	 * @param sourceFile full source file path
	 * @param destFile full path (or just directory) to move it to.
	 */
	public static String copy(String sourceFile, String destFile) {
    	byte[] xfer = new byte[4096];
    	int inBytes = 0;
		String sError = "";
		File sFile = new File(sourceFile);
		File dFile = new File(destFile);
	    FileInputStream inStream = null;
	    FileOutputStream outStream = null;
	    try {
			if (!dFile.exists()) {
				dFile.createNewFile();
			}
	    	inStream = new FileInputStream(sFile);
	    	outStream = new FileOutputStream(dFile);
    		inBytes = inStream.read(xfer);
	    	while (inBytes != -1) {
	    		outStream.write(xfer, 0, inBytes);
	    		inBytes = inStream.read(xfer);
	    	}
	    } catch(Exception e) {
	    	e.printStackTrace();
	    	sError = e.toString();
	    } finally {
	    	if (inStream != null) {
	    		try {
	    			inStream.close();
	    		} catch (IOException e) {
	    			;
	    		}
	    	}
	    	if (outStream != null) {
	    		try {
	    			outStream.close();
	    		} catch (IOException e) {
	    			;
	    		}
	    	}
	    }
		return sError;
	}

	/**
	 * Move a file to a specified directory.
	 * @param sourceFile full pathname of the original file
	 * @param destFile directory (or type) to move file to.
	 * @return true if it worked
	 */
	public static String move(String sourceFile, String destFile) {
		String sError = "";
		sError = copy(sourceFile, destFile);
		if (sError.equals("")) {
			sError = delete(sourceFile);
		}
		return sError;
	}

	/**
	 * Delete a file.
	 * @param sFile file to delete
	 */
	public static String delete(String sFile) {
		String sError = "";
		File oFile = new File(sFile);
		if (oFile.exists()) {
			oFile.delete();
			if (oFile.exists()) {
				oFile.deleteOnExit();
			}
		}
		return sError;
	}
	
	/**
	 * Add quotes to a file name if needed.  If not, we don't add them.  This means
	 * if a file name needs quotes because of spaces or other problems, we add them.
	 */
	public static String addNeededQuotes(String sFile) {
		String outFile = sFile;
		if (outFile.indexOf(" ") >= 0) {
			if (!outFile.endsWith("\"")) {
				outFile = outFile + "\"";
			}
			if (!outFile.startsWith("\"")) {
				outFile = "\"" + outFile;
			}
		}
		return outFile;
	}
}
