package com.hal.util;

import java.io.Serializable;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;


/**
 * Like a HashMap, but takes care of what Generics will take care of in later
 * Java versions.  Works only with Strings.  If an int is used instead of a String,
 * an overloaded method will take it and it will be converted to a String.
 * @author hal
 *
 */
public class StringHashMap implements Serializable {
	
	final static long serialVersionUID = 1L;

	HashMap myHash;
	
	/**
	 * Create a StringHashMap.
	 */
	public StringHashMap() {
		reset();
	}
	
	/**
	 * Clear all values in this StringHashMap
	 *
	 */
	public void reset() {
		myHash = new HashMap();
		return;
	}
	
	/**
	 * Put a value into the StringHashMap
	 * @param sName the key name
	 * @param sVal the value to put in for storage
	 */
	public void put(String sName, String sVal) {
		if (myHash.containsKey(sName))
			myHash.remove(sName);
		myHash.put(sName, sVal);		
		return;
	}
	
	/**
	 * Put a value in the StringHashMap.  The int will be converted to a String,
	 * which helps since some Java data types don't accept an int.
	 * @param sName key name
	 * @param iVal value to store
	 */
	public void put(String sName, int iVal) {
		put(sName, String.valueOf(iVal));		
		return;
	}
	
	/**
	 * Put a value in the StringHashMap.  The int will be converted to a String,
	 * which helps since some Java data types don't accept an int.
	 * @param iName key name as an integer
	 * @param sVal value to store
	 */
	public void put(int iName, String sVal) {
		put(String.valueOf(iName), sVal);		
		return;
	}
	
	/**
	 * Put a value in the StringHashMap.  The int will be converted to a String,
	 * which helps since some Java data types don't accept an int.
	 * @param iName key name as an integer
	 * @param iVal value to store
	 */
	public void put(int iName, int iVal) {
		put(String.valueOf(iName), String.valueOf(iVal));		
		return;
	}
	
	/**
	 * Get a value by specifying the key name
	 * @param sName the key for the value we want
	 * @return the value specified by the key
	 */
	public String get(String sName) {
		String sVal;
		sVal = (String) myHash.get(sName);
		return sVal;
	}
	
	/**
	 * Get a value by specifying the key name.  The int will be converted
	 * to a String then used as the key.
	 * @param iPoint The key for the value we want
	 * @return the value specified by the key.
	 */
	public String get(int iPoint) {
		String sName = String.valueOf(iPoint);
		String sVal = get(sName);
		return sVal;
	}
	
	/**
	 * Remove a value from the StringHashMap
	 * @param sName The key for the value we're removing
	 */
	public void remove(String sName) {
		if (myHash.containsKey(sName))
			myHash.remove(sName);
		return;
	}
	
	/**
	 * Get a sorted list of all the keys. 
	 * @return sorted list of keys
	 */
	public String[] keySet() {
		int i = 0;
		String[] aKeys;
		Set allKeys = myHash.keySet();
		Iterator iPoint = allKeys.iterator();
		aKeys = new String[allKeys.size()];
		while (iPoint.hasNext()) {
			Object nextItem = iPoint.next();
			if (nextItem != null) {
				aKeys[i++] = (String) nextItem;
			}
		}
		Arrays.sort(aKeys);
		return aKeys;
	}
	
	/**
	 * Get the number of keys with values.
	 * @return size of object.
	 */
	public int getLength() {
		int i = myHash.size();
		return i;
	}
	
	/**
	 * Get the number of the keys with values.
	 * @return size of object
	 */
	public int size() {
		return getLength();
	}
	
	/**
	 * Return true if we have a key matching the specified name
	 * @param keyName the key we're checking to see
	 * @return true if the key exists, false if not.
	 */
	public boolean hasKey(String keyName) {
		if (myHash.containsKey(keyName)) return true;
		return false;		
	}
	
	/**
	 * Return true if we have a key matching the specified name
	 * @param iPoint the key (in int form) we're checking to see
	 * @return true if the key exists, false if not.
	 */
	public boolean hasKey(int iPoint) {
		if (myHash.containsKey(String.valueOf(iPoint))) return true;
		return false;		
	}
}
